// Server configuration
const SERVER_URL = 'https://divinge.app';

// DOM elements
const statusDot = document.getElementById('statusDot');
const statusText = document.getElementById('statusText');
const cookiePreview = document.getElementById('cookiePreview');
const syncBtn = document.getElementById('syncBtn');
const result = document.getElementById('result');
const tradeSection = document.getElementById('tradeSection');
const detectedQuery = document.getElementById('detectedQuery');
const queryNameInput = document.getElementById('queryName');
const registerBtn = document.getElementById('registerBtn');
const syncTokenInput = document.getElementById('syncToken');
const saveConfigBtn = document.getElementById('saveConfigBtn');
const connectionStatus = document.getElementById('connectionStatus');
const connDot = document.getElementById('connDot');
const connText = document.getElementById('connText');

let currentCookies = null;
let detectedTrade = null;
let syncToken = '';
let pollInterval = null;
let isPolling = false;

// Initialize on load
document.addEventListener('DOMContentLoaded', async () => {
  await loadConfig();
  checkCookies();
  checkTradePage();
});

// Load sync token from storage
async function loadConfig() {
  const data = await chrome.storage.local.get('syncToken');
  syncToken = data.syncToken || '';

  if (syncToken) {
    syncTokenInput.value = '••••••••••••••••';
    syncTokenInput.type = 'password';
    await checkConnection();
  } else {
    updateConnectionStatus(false, 'No token configured');
    syncBtn.disabled = true;
  }
}

// Save sync token
saveConfigBtn.addEventListener('click', async () => {
  const token = syncTokenInput.value.trim();

  // Don't save masked value
  if (token === '••••••••••••••••') {
    showResult('Token already saved', 'success');
    return;
  }

  if (!token) {
    showResult('Please enter your sync token', 'error');
    return;
  }

  syncToken = token;
  await chrome.storage.local.set({ syncToken: token });

  // Mask the input
  syncTokenInput.value = '••••••••••••••••';
  syncTokenInput.type = 'password';

  // Verify token with server
  await checkConnection();

  // Re-check cookies
  checkCookies();
});

// Check connection to server with token
async function checkConnection() {
  updateConnectionStatus(null, 'Connecting...');

  try {
    const response = await fetch(`${SERVER_URL}/api/config`, {
      headers: {
        'Authorization': `Bearer ${syncToken}`
      }
    });

    if (response.ok) {
      const data = await response.json();
      updateConnectionStatus(true, `Connected as ${data.user?.name || 'User'}`);
      syncBtn.disabled = !currentCookies?.poesessid;
      return true;
    } else if (response.status === 401) {
      updateConnectionStatus(false, 'Invalid token');
      syncBtn.disabled = true;
      return false;
    } else {
      updateConnectionStatus(false, 'Connection failed');
      syncBtn.disabled = true;
      return false;
    }
  } catch (err) {
    updateConnectionStatus(false, 'Cannot reach server');
    syncBtn.disabled = true;
    return false;
  }
}

// Update connection status display
function updateConnectionStatus(connected, message) {
  connText.textContent = message;

  if (connected === true) {
    connDot.className = 'dot connected';
    connectionStatus.className = 'connection-status connected';
  } else if (connected === false) {
    connDot.className = 'dot missing';
    connectionStatus.className = 'connection-status';
  } else {
    // null = pending
    connDot.className = 'dot';
    connectionStatus.className = 'connection-status';
  }
}

// Sync button - handles the entire flow
syncBtn.addEventListener('click', handleSyncClick);

// Register trade button
registerBtn.addEventListener('click', registerTrade);

// Main sync flow handler
async function handleSyncClick() {
  if (!syncToken) {
    showResult('Please configure your sync token first', 'error');
    return;
  }

  // Re-check cookies first
  await checkCookies();

  if (currentCookies?.poesessid) {
    // Cookie exists - sync immediately
    await syncCookies();
  } else {
    // No cookie - open PoE and start polling
    await startPollingFlow();
  }
}

// Open PoE and poll for cookie
async function startPollingFlow() {
  if (isPolling) return;

  isPolling = true;
  syncBtn.disabled = true;
  syncBtn.textContent = 'Waiting for login...';

  // Update status to show we're waiting
  statusDot.className = 'dot polling';
  statusText.textContent = 'Waiting...';
  cookiePreview.textContent = 'Log into pathofexile.com in the new tab';
  showResult('Opening Path of Exile - please log in', 'info');

  // Open PoE trade page
  await chrome.tabs.create({ url: 'https://www.pathofexile.com/trade2/search/poe2/Standard' });

  // Start polling for cookie
  let attempts = 0;
  const maxAttempts = 60; // 2 minutes max

  pollInterval = setInterval(async () => {
    attempts++;

    try {
      const cookies = await chrome.runtime.sendMessage({ action: 'getCookies' });

      if (cookies.poesessid) {
        // Cookie found! Stop polling and sync
        stopPolling();
        currentCookies = cookies;

        // Update UI to show found
        statusDot.className = 'dot found';
        statusText.textContent = 'Found';
        cookiePreview.textContent = `...${cookies.poesessid.slice(-12)}`;

        // Auto-sync
        showResult('Cookie found! Syncing...', 'success');
        await syncCookies();
        return;
      }
    } catch (err) {
      console.error('Poll error:', err);
    }

    // Update waiting indicator
    const dots = '.'.repeat((attempts % 3) + 1);
    syncBtn.textContent = `Waiting for login${dots}`;

    // Timeout after max attempts
    if (attempts >= maxAttempts) {
      stopPolling();
      showResult('Timed out waiting for login. Try again.', 'error');
      syncBtn.disabled = false;
      syncBtn.textContent = 'Sync Cookies';
      checkCookies();
    }
  }, 2000); // Poll every 2 seconds
}

// Stop polling
function stopPolling() {
  isPolling = false;
  if (pollInterval) {
    clearInterval(pollInterval);
    pollInterval = null;
  }
}

// Check if current tab is a PoE trade search page
async function checkTradePage() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab?.url) return;

    // Match URLs like /trade2/search/poe2/League/QueryID
    const match = tab.url.match(/\/trade2\/search\/poe2\/([^/]+)\/([^/?]+)/);
    if (match) {
      const league = decodeURIComponent(match[1]);
      const queryId = match[2];

      detectedTrade = { queryId, league };
      detectedQuery.textContent = `${league} / ${queryId}`;
      tradeSection.style.display = 'block';
    }
  } catch (err) {
    console.error('Failed to check trade page:', err);
  }
}

// Register the detected trade as a query
async function registerTrade() {
  if (!detectedTrade) return;
  if (!syncToken) {
    showResult('Please configure your sync token first', 'error');
    return;
  }

  registerBtn.disabled = true;
  registerBtn.textContent = 'Registering...';

  try {
    const response = await fetch(`${SERVER_URL}/api/queries`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${syncToken}`
      },
      body: JSON.stringify({
        id: detectedTrade.queryId,
        name: queryNameInput.value.trim() || detectedTrade.queryId,
        league: detectedTrade.league,
      }),
    });

    const data = await response.json();

    if (response.ok && data.success) {
      showResult(`Registered: ${queryNameInput.value.trim() || detectedTrade.queryId}`, 'success');
      registerBtn.textContent = 'Registered!';
      setTimeout(() => window.close(), 1500);
    } else {
      showResult(data.error || 'Registration failed', 'error');
      registerBtn.disabled = false;
      registerBtn.textContent = 'Register Trade';
    }
  } catch (err) {
    showResult('Cannot connect to server', 'error');
    registerBtn.disabled = false;
    registerBtn.textContent = 'Register Trade';
  }
}

async function checkCookies() {
  try {
    const cookies = await chrome.runtime.sendMessage({ action: 'getCookies' });
    currentCookies = cookies;

    if (cookies.poesessid) {
      statusDot.className = 'dot found';
      statusText.textContent = 'Found';
      cookiePreview.textContent = `...${cookies.poesessid.slice(-12)}`;
      // Only enable sync if we also have a valid token
      syncBtn.disabled = !syncToken;
    } else {
      statusDot.className = 'dot missing';
      statusText.textContent = 'Not logged in';
      cookiePreview.textContent = 'Click sync to log into PoE';
      // Enable button - it will handle opening PoE
      syncBtn.disabled = !syncToken;
    }
  } catch (err) {
    statusDot.className = 'dot missing';
    statusText.textContent = 'Error';
    cookiePreview.textContent = err.message;
  }
}

async function syncCookies() {
  if (!currentCookies?.poesessid) {
    showResult('No POESESSID found', 'error');
    return;
  }

  if (!syncToken) {
    showResult('Please configure your sync token first', 'error');
    return;
  }

  syncBtn.disabled = true;
  syncBtn.textContent = 'Syncing...';
  result.className = 'result';

  try {
    const response = await fetch(`${SERVER_URL}/api/cookie`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${syncToken}`
      },
      body: JSON.stringify({
        poesessid: currentCookies.poesessid,
        cfClearance: currentCookies.cfClearance,
      }),
    });

    const data = await response.json();

    if (response.ok && data.success) {
      showResult('Synced! Returning to dashboard...', 'success');

      // Close any pathofexile.com tabs
      const tabs = await chrome.tabs.query({ url: '*://*.pathofexile.com/*' });
      for (const tab of tabs) {
        chrome.tabs.remove(tab.id);
      }

      // Open/focus the dashboard
      const dashboardTabs = await chrome.tabs.query({ url: `${SERVER_URL}/*` });
      if (dashboardTabs.length > 0) {
        // Focus existing dashboard tab
        await chrome.tabs.update(dashboardTabs[0].id, { active: true });
      } else {
        // Open new dashboard tab
        await chrome.tabs.create({ url: `${SERVER_URL}/app` });
      }

      // Close the popup after a moment
      setTimeout(() => window.close(), 1000);
    } else {
      showResult(data.error || 'Sync failed', 'error');
      syncBtn.disabled = false;
      syncBtn.textContent = 'Sync Cookies';
    }
  } catch (err) {
    showResult('Cannot connect to server. Is it running?', 'error');
    syncBtn.disabled = false;
    syncBtn.textContent = 'Sync Cookies';
  }
}

function showResult(message, type) {
  result.textContent = message;
  result.className = `result ${type}`;
}
